using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;

namespace BreadAndCheese {

#region Game Sprites

    public class BaseSprite {
        protected Texture2D spriteTexture;
        protected Rectangle spriteRectangle;

        /// <summary>
        /// aduje tekstur uywan w ramach danego sprajtu.
        /// </summary>
        /// <param name="inSpriteTexture">Uywana tekstura.</param>
        public void LoadTexture(Texture2D inSpriteTexture) {
            spriteTexture = inSpriteTexture;
        }

        /// <summary>
        /// aduje prostokt wyznaczajcy wymiary na potrzeby 
        /// operacji rysowania tego sprajtu ta.
        /// </summary>
        /// <param name="inSpriteRectangle">Prostokt, ktry ma zosta uyty.</param>
        public void SetRectangle(Rectangle inSpriteRectangle) {
            spriteRectangle = inSpriteRectangle;
        }

        /// <summary>
        /// Rysuje tekstur na podstawie prostokta okrelajcego jej wymiary.
        /// </summary>
        /// <param name="spriteBatch">Obiekt klasy SpriteBatch, ktry ma zosta uyty
        /// przez operacj rysowania.</param>
        public virtual void Draw(SpriteBatch spriteBatch) {
            spriteBatch.Draw(spriteTexture, spriteRectangle, Color.White);
        }

        /// <summary>
        /// Aktualizuje zachowanie dla tytuu. Sprajt bazowy nie zawiera adnego
        /// zachowania aktualizacji, ale pozostae sprajty mog zawiera takie zachowania.
        /// </summary>
        /// <param name="game">Gra, ktrej przebieg ma by kontrolowany.</param>
        public virtual void Update(BreadAndCheeseGame game) {
        }
    }

    public class TitleSprite : BaseSprite {
        /// <summary>
        /// Aktualizuje zachowanie dla tytuu. Metoda testuje stan przycisku A i uruchamia gr,
        /// jeli ten przycisk jest wcinity.
        /// </summary>
        /// <param name="game">Gra, ktrej przebieg ma by kontrolowany.</param>
        public override void Update(BreadAndCheeseGame game) {
            if (game.GamePad1.Buttons.A == ButtonState.Pressed) {
                game.StartGame();
            }
        }
    }

    public class MovingSprite : BaseSprite {
        protected float x;
        protected float y;
        protected float xSpeed;
        protected float ySpeed;

        protected float minDisplayX;
        protected float maxDisplayX;
        protected float minDisplayY;
        protected float maxDisplayY;

        /// <summary>
        /// Metoda rozpoczyna gr. Wyznacza rozmiary wszystkich obiektw na 
        /// ekranie i umieszcza na pocztkowych pozycjach.
        /// </summary>
        /// <param name="widthFactor">Rozmiar obiektu w grze wyraony jako uamek szerokoci
        /// ekranu.</param>
        /// <param name="ticksToCrossScreen">Szybko poruszania si obiektu wyraona w liczbie taktw (1/60 sekundy) 
        /// potrzebnych do pokonania caego ekranu.</param>
        /// <param name="inMinDisplayX">minimalna warto X</param>
        /// <param name="inMaxDisplayX">maksymalna warto X</param>
        /// <param name="inMinDisplayY">minimalna warto Y</param>
        /// <param name="inMaxDisplayY">maksymalna warto Y</param>
        /// <param name="initialX">pocztkowe pooenie sprajtu w poziomie (X)</param>
        /// <param name="initialY">pocztkowe pooenie sprajtu w pionie (Y)</param>

        public virtual void StartGame(
                float widthFactor,
                float ticksToCrossScreen,
                float inMinDisplayX,
                float inMaxDisplayX,
                float inMinDisplayY,
                float inMaxDisplayY,
                float initialX,
                float initialY) {
            minDisplayX = inMinDisplayX;
            minDisplayY = inMinDisplayY;
            maxDisplayX = inMaxDisplayX;
            maxDisplayY = inMaxDisplayY;

            float displayWidth = maxDisplayX - minDisplayX;

            spriteRectangle.Width = (int) ((displayWidth * widthFactor) + 0.5f);
            float aspectRatio =
                    (float) spriteTexture.Width / spriteTexture.Height;
            spriteRectangle.Height =
                    (int) ((spriteRectangle.Width / aspectRatio) + 0.5f);
            x = initialX;
            y = initialY;
            xSpeed = displayWidth / ticksToCrossScreen;
            ySpeed = xSpeed;
        }

        /// <summary>
        /// Sprawdza ewentualne kolizje pomidzy danym sprajtem a pozostaymi obiektami.
        /// </summary>
        /// <param name="ballPosition">Prostokt reprezentujcy pooenie innego obiektu.</param>
        /// <returns>true, jeli sprajt koliduje z danym obiektem</returns>
        public virtual bool CheckCollision(Rectangle target) {
            return spriteRectangle.Intersects(target);
        }
    }

    public class BatSprite : MovingSprite {
        /// <summary>
        /// Aktualizuje pooenie paki zalenie od stanu pada.
        /// </summary>
        public override void Update(BreadAndCheeseGame game) {
            x = x + (xSpeed * game.GamePad1.ThumbSticks.Left.X);
            y = y - (ySpeed * game.GamePad1.ThumbSticks.Left.Y);
            spriteRectangle.X = (int) x;
            spriteRectangle.Y = (int) y;
        }
    }

    public class BallSprite : MovingSprite {
        /// <summary>
        /// Aktualizuje pooenie piki. Obsuguje kolizje z pak
        /// i celami. 
        /// </summary>
        /// <param name="game">Gra, w ktrej wystpuje ta pika</param>
        public override void Update(BreadAndCheeseGame game) {
            x = x + xSpeed;
            y = y + ySpeed;

            spriteRectangle.X = (int) (x + 0.5f);
            spriteRectangle.Y = (int) (y + 0.5f);

            if (game.BreadBat.CheckCollision(spriteRectangle)) {
                // Pika trafia w pak.
                ySpeed = ySpeed * -1;
            }

            if (x + spriteRectangle.Width >= maxDisplayX) {
                // Pika zostaa trafiona z prawej strony.
                xSpeed = Math.Abs(xSpeed) * -1;
            }

            if (x <= minDisplayX) {
                // Pika zostaa trafiona z lewej strony.
                xSpeed = Math.Abs(xSpeed);
            }

            if (y + spriteRectangle.Height >= maxDisplayY) {
                // Pika zderzya si z doln krawdzi. Powoduje utrat ycia.
                ySpeed = Math.Abs(ySpeed) * -1;
                game.LoseLife();
            }

            if (y <= minDisplayY) {
                // Pika uderzya w grn cz ekranu.
                ySpeed = Math.Abs(ySpeed);
            }

            if (game.TomatoTargets.CheckCollision(spriteRectangle)) {
                game.UpdateScore(20);
                ySpeed = ySpeed * -1;
            }
        }
    }

    public class TargetRowSprite : BaseSprite {
        private int numberOfTargets;
        private float targetWidth;
        private float targetHeight;
        private float targetStepFactor;
        private float targetHeightLimit;
        private Rectangle[] Targets;
        private bool[] TargetVisibility;

        private float minDisplayX;
        private float maxDisplayX;
        private float minDisplayY;
        private float maxDisplayY;
        private float displayHeight;
        private float displayWidth;

        /// <summary>
        /// Rozpoczyna gr. Konfiguruje cele i przygotowuje je do uycia.
        /// </summary>
        /// <param name="inNumberOfTargets">Liczba celw na ekranie.</param>
        /// <param name="inTargetStepFactor">Cz ekranu, o ktr cele zostan przesunite w d
        /// po zniszczeniu caego rzdu.</param>
        /// <param name="inMinDisplayX"></param>
        /// <param name="inMaxDisplayX"></param>
        /// <param name="inMinDisplayY"></param>
        /// <param name="inMaxDisplayY"></param>
        public void StartGame(
                int inNumberOfTargets,
                float inTargetStepFactor,
                float inMinDisplayX,
                float inMaxDisplayX,
                float inMinDisplayY,
                float inMaxDisplayY) {
            numberOfTargets = inNumberOfTargets;
            targetStepFactor = inTargetStepFactor;

            minDisplayX = inMinDisplayX;
            minDisplayY = inMinDisplayY;
            maxDisplayX = inMaxDisplayX;
            maxDisplayY = inMaxDisplayY;

            displayWidth = maxDisplayX - minDisplayX;
            displayHeight = maxDisplayY - minDisplayY;

            targetWidth = displayWidth / numberOfTargets;

            float aspectRatio = (float) spriteTexture.Width / spriteTexture.Height;

            targetHeight = targetWidth / aspectRatio;

            targetHeightLimit = minDisplayY + ((maxDisplayY - minDisplayY) / 2);

            Targets = new Rectangle[numberOfTargets];
            TargetVisibility = new bool[numberOfTargets];

            for (int i = 0; i < numberOfTargets; i++) {
                Targets[i].Width = (int) targetWidth;
                Targets[i].Height = (int) targetHeight;

                Targets[i].Y = (int) targetHeight;
                Targets[i].X = (int) (minDisplayX + (i * targetWidth) + 0.5f);

                TargetVisibility[i] = true;
            }
        }

        /// <summary>
        /// Aktualizuje sposb wywietlania ballPosition. Jeli wszystkie cele zostay zniszczone
        /// pooenie rzdu celw jest zmieniane, a same cele s ponownie wywietlane na ekranie.
        /// </summary>
        /// <param name="game">Gra zawierajca dany rzd ballPosition. Ten parametr nie jest obecnie uywany
        /// ale moe zosta uyty w przyszoci, aby umoliwi ballPosition przyspieszanie ruchu paki
        /// po kadym zniszczeniu caego rzdu celw.</param>
        public override void Update(BreadAndCheeseGame game) {
            for (int i = 0; i < numberOfTargets; i++) {
                if (TargetVisibility[i]) {
                    // Zwraca sterowanie w razie znalezienia widocznego obiektu ballPosition.
                    return;
                }
            }

            // Jeli sterowanie dotaro w to miejsce, nie ma widocznych pomidorw.

            // Pooenie obiektu ballPosition na ekranie jest obniane.
            targetHeight = targetHeight + (displayHeight * targetStepFactor);

            // Sprawdza, czy osignito doln granic rysowania celw.
            if (targetHeight > targetHeightLimit) {
                targetHeight = minDisplayY;
            }

            // Przywraca ustawienia wszystkich celw.
            for (int i = 0; i < numberOfTargets; i++) {
                Targets[i].Y = (int) targetHeight;
                TargetVisibility[i] = true;
            }
        }

        /// <summary>
        /// Rysuje wszystkie widoczne cele.
        /// </summary>
        /// <param name="spriteBatch">Obiekt klasy SpriteBatch, ktry ma zosta uyty podczas rysowania.</param>
        public override void Draw(SpriteBatch spriteBatch) {
            for (int i = 0; i < numberOfTargets; i++) {
                if (TargetVisibility[i]) {
                    spriteBatch.Draw(spriteTexture, Targets[i], Color.White);
                }
            }
        }

        /// <summary>
        /// Sprawdza, czy pika nie koliduje z obiektem ballPosition. Jeli tak,
        /// obiekt ballPosition zostaje uznany za zniszczony.
        /// </summary>
        /// <param name="ballPosition">Pooenie piki.</param>
        /// <returns>true, jeli miaa miejsce kolizja i jeli cel zosta usunity.</returns>
        public bool CheckCollision(Rectangle ballPosition) {
            for (int i = 0; i < numberOfTargets; i++) {
                if (TargetVisibility[i]) {
                    // Uzyskuje cel uczestniczcy w kolizji.
                    if (Targets[i].Intersects(ballPosition)) {
                        // Cel jest niszczony.
                        TargetVisibility[i] = false;
                        // Zwraca warto reprezentujc wystpienie kolizji.
                        return true;
                    }
                }
            }
            return false;
        }
    }

#endregion

    /// <summary>
    /// To jest gwny typ Twojej gry.
    /// </summary>
    public class BreadAndCheeseGame : Microsoft.Xna.Framework.Game {
        GraphicsDeviceManager graphics;
        SpriteBatch spriteBatch;

        // wiat gry
        public GamePadState GamePad1;

        public BatSprite BreadBat;
        public BallSprite CheeseBall;
        public TargetRowSprite TomatoTargets;
        public TitleSprite Title;
        public BaseSprite Background;

#region Wartoci reprezentujce wymiary ekranu

        int displayWidth;
        int displayHeight;
        float overScanPercentage = 10.0f;
        float minDisplayX;
        float maxDisplayX;
        float minDisplayY;
        float maxDisplayY;

        private void setScreenSizes() {
            displayWidth = graphics.GraphicsDevice.Viewport.Width;
            displayHeight = graphics.GraphicsDevice.Viewport.Height;
            float xOverscanMargin = getPercentage(overScanPercentage, displayWidth) / 2.0f;
            float yOverscanMargin = getPercentage(overScanPercentage, displayHeight) / 2.0f;

            minDisplayX = xOverscanMargin;
            minDisplayY = yOverscanMargin;

            maxDisplayX = displayWidth - xOverscanMargin;
            maxDisplayY = displayHeight - yOverscanMargin;
        }


#endregion

#region Metody pomocnicze

        /// <summary>
        /// Oblicza wartoci procentowe
        /// </summary>
        /// <param name="percentage">procent, ktry ma zosta wyznaczony</param>
        /// <param name="inputValue">warto do przeliczenia</param>
        /// <returns>warto wynikowa</returns>

        float getPercentage(float percentage, float inputValue) {
            return (inputValue * percentage) / 100;
        }

#endregion

#region Rysowanie tekstu

        SpriteFont font;

        private void loadFont() {
            font = Content.Load<SpriteFont > ("SpriteFont1");
        }

        /// <summary>
        /// Rysuje tekst na ekranie
        /// </summary>
        /// <param name="text">tekst do wywietlenia</param>
        /// <param name="textColor">kolor tekstu</param>
        /// <param name="x">lewa krawd tekstu</param>
        /// <param name="y">grna krawd tekstu</param>

        void drawText(string text, Color textColor, float x, float y) {
            int layer;
            Vector2 textVector = new Vector2(x, y);

            // Rysuje cie
            Color backColor = new Color(0, 0, 0, 10);
            for (layer = 0; layer < 10; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X++;
                textVector.Y++;
            }

            // Rysuje podstaw znakw
            backColor = new Color(190, 190, 190);
            for (layer = 0; layer < 5; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X++;
                textVector.Y++;
            }

            // Rysuje przedni cz znakw
            spriteBatch.DrawString(font, text, textVector, textColor);
        }

#endregion

#region Wywietlanie wyniku

        void drawScore() {
            drawText("Punkty: " + score + " ycia: " + lives, Color.Blue, minDisplayX, maxDisplayY - 50);
        }

        void drawHighScore() {
            drawText("Najlepszy wynik: " + highScore + "Nacinij A, aby rozpocz", Color.Blue, minDisplayX, minDisplayY);
        }

#endregion

#region Zarzdzanie gr i punktacj

        int score;
        int lives;
        int highScore;

        public

               enum GameState {
            titleScreen,
            playingGame
        }

        GameState state = GameState.titleScreen;

        public void StartGame() {
            score = 0;
            lives = 3;

            CheeseBall.StartGame(
                    0.07f, // szeroko sera wynosi 0,07 szerokoci ekranu
                    200, // ser potrzebuje 200 taktw na pokonanie caego ekranu
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    displayWidth / 4, // jedna czwarta szerokoci ekranu
                    displayHeight / 4); // jedna czwarta wysokoci ekranu od grnej krawdzi

            BreadBat.StartGame(
                    0.166f, // szeroko chleba wynosi 0,166 szerokoci ekranu
                    150, // pokonanie ekranu zajmuje 150 taktw
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    displayWidth / 2, // na pocztku znajduje si na rodku ekranu (w poziomie)
                    displayHeight / 2); // na pocztku znajduje si na rodku ekranu (w pionie)

            TomatoTargets.StartGame(
                    20,
                    0.1f,
                    minDisplayX,
                    maxDisplayX,
                    minDisplayY,
                    maxDisplayY);

            state = GameState.playingGame;
        }

        void gameOver() {
            if (score > highScore) {
                highScore = score;
            }
            state = GameState.titleScreen;
        }

        public void LoseLife() {
            lives--;
            if (lives == 0) {
                gameOver();
            }
        }

        public void UpdateScore(int update) {
            score = score + update;
        }

        public int GetScore() {
            return score;
        }

        public int GetLives() {
            return lives;
        }

        public GameState GetState() {
            return state;
        }

#endregion

        public BreadAndCheeseGame() {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";
        }

        /// <summary>
        /// Umoliwia ewentualn inicjalizacj przed uruchomieniem waciwej gry.
        /// W tym miejscu mona odnale wszystkie wymagane zasoby i zaadowa treci
        /// related content. Wywoanie metody base.Initialize spowoduje przeszukanie wszystkich komponentw
        /// i ich inicjalizacj.
        /// </summary>
        protected override void Initialize() {
            setScreenSizes();

            BreadBat = new BatSprite();
            CheeseBall = new BallSprite();
            TomatoTargets = new TargetRowSprite();
            Title = new TitleSprite();
            Background = new BaseSprite();

            Title.SetRectangle(
                    new Rectangle(
                    (int) minDisplayX, (int) minDisplayY,
                    (int) (maxDisplayX - minDisplayX),
                    (int) (maxDisplayY - minDisplayY)
                    ));

            Background.SetRectangle(
                    new Rectangle(
                    (int) minDisplayX, (int) minDisplayY,
                    (int) (maxDisplayX - minDisplayX),
                    (int) (maxDisplayY - minDisplayY)
                    ));

            base.Initialize();
        }

        /// <summary>
        /// Metoda LoadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu adujcego tre.
        /// </summary>
        protected override void LoadContent() {
            // Tworzy nowy obiekt klasy SpriteBatch, ktrego mona uywa do rysowania tekstur.
            spriteBatch = new SpriteBatch(GraphicsDevice);

            BreadBat.LoadTexture(Content.Load<Texture2D > ("Images/Bread"));
            CheeseBall.LoadTexture(Content.Load<Texture2D > ("Images/Cheese"));
            TomatoTargets.LoadTexture(Content.Load<Texture2D > ("Images/Tomato"));
            Title.LoadTexture(Content.Load<Texture2D > ("Images/Title"));
            Background.LoadTexture(Content.Load<Texture2D > ("Images/Background"));
            loadFont();
        }

        /// <summary>
        /// Metoda UnloadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu usuwajcego ca tre z pamici.
        /// </summary>
        protected override void UnloadContent() {
            // TODO: Naley usun ca tre, ktr nie zarzdza ContentManager.
        }

        /// <summary>
        /// Umoliwia grze wykonywanie logiki zwizanej z aktualizacj wiata gry,
        /// sprawdzaniem kolizji, pobieraniem danych wejciowych czy odtwarzaniem dwikw.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Update(GameTime gameTime) {
            GamePad1 = GamePad.GetState(PlayerIndex.One);

            if (GamePad1.Buttons.Back == ButtonState.Pressed)
                this.Exit();

            switch (state) {
                case GameState.titleScreen:
                    Title.Update(this);
                    break;
                case GameState.playingGame:
                    BreadBat.Update(this);
                    CheeseBall.Update(this);
                    TomatoTargets.Update(this);
                    break;
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// Ta metoda jest wywoywana w momencie, w ktrym gra musi narysowa swj wiat.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Draw(GameTime gameTime) {
            GraphicsDevice.Clear(Color.CornflowerBlue);

            spriteBatch.Begin();

            switch (state) {
                case GameState.titleScreen:
                    Title.Draw(spriteBatch);
                    drawHighScore();
                    break;
                case GameState.playingGame:
                    Background.Draw(spriteBatch);
                    BreadBat.Draw(spriteBatch);
                    CheeseBall.Draw(spriteBatch);
                    TomatoTargets.Draw(spriteBatch);
                    drawScore();
                    break;
            }

            spriteBatch.End();

            base.Draw(gameTime);
        }
    }
}
